"use strict";

const _ = require('lodash');
/**
 * Appends the provided context(s) to the list of contexts that will be used by the NLU Engine
 * for the next messages for that chat session.
 *
 * The TTL (Time-To-Live) represents how long the contexts will be valid before they are automatically removed.
 * For example, the default value of `1` will listen for that context only once (the next time the user speaks).
 *
 * If a context was already present in the list, the higher TTL will win.
 * To force override a specific context, use the `removeContext` action before this action.
 *
 * This method is contextual to the current user and chat session.
 *
 * You can specify more than one context by separating them with a comma.
 *
 * @title Append Context
 * @category NLU
 * @author Botpress, Inc.
 * @param {string} contexts - Comma-separated list of contexts
 * @param {string} [ttl=1] - Time-To-Live of the context in number of dialog turns. Put `0` to disable expiry.
 */


const appendContext = (contexts, ttl) => {
  const existing = event.state.session.nluContexts || [];
  const add = contexts.trim().split(',');
  const merged = [...existing, ...add.map(x => ({
    context: x,
    ttl: isNaN(Number(ttl)) ? 1000 : Number(ttl)
  }))];
  const final = [];
  const visited = {};

  for (const ctx of merged) {
    if (visited[ctx.context]) {
      continue;
    }

    final.push(_.chain(merged).filter(x => x.context === ctx.context).orderBy('ttl', 'desc').first().value());
    visited[ctx.context] = true;
  }

  event.state.session.nluContexts = final;
};

return appendContext(args.contexts, args.ttl);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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