"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupBot = setupBot;
exports.setupMiddleware = setupMiddleware;

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

var _telegraf = require("telegraf");

var _extra = _interopRequireDefault(require("telegraf/extra"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const outgoingTypes = ['text', 'typing', 'login_prompt', 'carousel'];

async function setupBot(bp, botId, clients) {
  const client = clients[botId];

  const send = (ctx, args) => bp.events.sendEvent(bp.IO.Event({
    botId,
    channel: 'telegram',
    direction: 'incoming',
    payload: ctx.message,
    preview: ctx.message.text,
    threadId: ctx.chat.id.toString(),
    target: ctx.from.id.toString(),
    ...args
  }));

  client.start(ctx => send(ctx, {
    type: 'start'
  }));
  client.help(ctx => send(ctx, {
    type: 'help'
  }));
  client.on('message', ctx => send(ctx, {
    type: 'message'
  }));
  client.on('callback_query', ctx => send(ctx, {
    type: 'callback'
  })); // TODO We don't support understanding and accepting more complex stuff from users such as files, audio etc
}

async function setupMiddleware(bp, clients) {
  bp.events.registerMiddleware({
    description: 'Sends out messages that targets platform = telegram.' + ' This middleware should be placed at the end as it swallows events once sent.',
    direction: 'outgoing',
    handler: outgoingHandler,
    name: 'telegram.sendMessages',
    order: 100
  });

  async function outgoingHandler(event, next) {
    if (event.channel !== 'telegram') {
      return next();
    }

    const client = clients[event.botId];

    if (!client) {
      return next();
    }

    const messageType = event.type === 'default' ? 'text' : event.type;
    const chatId = event.threadId || event.target;

    if (!_lodash.default.includes(outgoingTypes, messageType)) {
      return next(new Error('Unsupported event type: ' + event.type));
    }

    if (messageType === 'typing') {
      await sendTyping(event, client, chatId);
    } else if (messageType === 'text') {
      await sendTextMessage(event, client, chatId);
    } else if (messageType === 'carousel') {
      await sendCarousel(event, client, chatId);
    } else {
      // TODO We don't support sending files, location requests (and probably more) yet
      throw new Error(`Message type "${messageType}" not implemented yet`);
    }

    next(undefined, false);
  }
}

async function sendCarousel(event, client, chatId) {
  if (event.payload.elements && event.payload.elements.length) {
    const {
      title,
      picture,
      subtitle
    } = event.payload.elements[0];
    const buttons = event.payload.elements.map(x => x.buttons);

    if (picture) {
      await client.telegram.sendChatAction(chatId, 'upload_photo');
      await client.telegram.sendPhoto(chatId, {
        url: picture,
        filename: _path.default.basename(picture)
      });
    }

    const keyboard = keyboardButtons(buttons);
    await client.telegram.sendMessage(chatId, `*${title}*\n${subtitle}`, _extra.default.markdown(true).markup(_telegraf.Markup.inlineKeyboard(keyboard)));
  }
}

async function sendTextMessage(event, client, chatId) {
  const keyboard = _telegraf.Markup.keyboard(keyboardButtons(event.payload.quick_replies));

  if (event.payload.markdown != false) {
    // Attempt at sending with markdown first, fallback to regular text on failure
    await client.telegram.sendMessage(chatId, event.preview, _extra.default.markdown(true).markup({ ...keyboard,
      one_time_keyboard: true
    })).catch(() => client.telegram.sendMessage(chatId, event.preview, _extra.default.markdown(false).markup({ ...keyboard,
      one_time_keyboard: true
    })));
  } else {
    await client.telegram.sendMessage(chatId, event.preview, _extra.default.markdown(false).markup({ ...keyboard,
      one_time_keyboard: true
    }));
  }
}

async function sendTyping(event, client, chatId) {
  const typing = parseTyping(event.payload.value);
  await client.telegram.sendChatAction(chatId, 'typing');
  await Promise.delay(typing);
}

function parseTyping(typing) {
  if (isNaN(typing)) {
    return 1000;
  }

  return Math.max(typing, 500);
}

function keyboardButtons(arr) {
  if (!arr || !arr.length) {
    return undefined;
  }

  const rows = arr[0].length ? arr : [arr];
  return rows.map(row => row.map(x => {
    if (x.url) {
      return _telegraf.Markup.urlButton(x.title, x.url);
    }

    return _telegraf.Markup.callbackButton(x.title, x.payload);
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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