"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var _default = ({
  bp,
  botId
}) => {
  const graphs = [];
  const knex = bp.database;

  async function update(name, operation, value, racing = false) {
    if (!_lodash.default.isString(name)) {
      throw new Error('Invalid name, expected a string');
    }

    const today = (0, _moment.default)().format('YYYY-MM-DD');
    name = name.toLowerCase().trim();

    if (!name.includes('~')) {
      name += '~';
    }

    const result = await knex('analytics_custom').where('date', today).andWhere('botId', botId).andWhere('name', name).update('count', operation).then();

    if (result == 0 && !racing) {
      await knex('analytics_custom').insert({
        botId,
        name: name,
        date: today,
        count: value
      }).catch(err => {
        return update(name, operation, value, true);
      });
    }
  }

  async function increment(name, count = 1) {
    if (!_lodash.default.isNumber(count)) {
      throw new Error('Invalid count increment, expected a valid number');
    }

    const countQuery = count < 0 ? 'count - ' + Math.abs(count) : 'count + ' + Math.abs(count);
    return update(name, knex.raw(countQuery), count);
  }

  async function decrement(name, count = 1) {
    return increment(name, count * -1);
  }

  async function set(name, count = 1) {
    return update(name, count, count);
  } // { name, type, description, variables }


  function addGraph(graph) {
    if (!_lodash.default.includes(['count', 'countUniq', 'percent', 'piechart'], graph.type)) {
      throw new Error('Unknown graph of type ' + graph.type);
    }

    graphs.push(graph);
  }

  const countUniqRecords = async (from, to, variable) => {
    const uniqRecordsQuery = function () {
      this.select(knex.raw('distinct name')).from('analytics_custom').where('date', '>=', from).andWhere('botId', botId).andWhere('date', '<=', to).andWhere('name', 'LIKE', variable + '~%').as('t1');
    };

    const {
      count
    } = await knex.count('*').from(uniqRecordsQuery).first();
    return count;
  };

  const getters = {
    count: async function (graph, from, to) {
      const variable = _lodash.default.first(graph.variables);

      const rows = await knex('analytics_custom').select(['date', knex.raw('sum(count) as count')]).where('date', '>=', from).andWhere('botId', botId).andWhere('date', '<=', to).andWhere(function () {
        if (variable.indexOf('~') > -1) {
          return this.where('name', 'LIKE', variable);
        } else {
          return this.where('name', 'LIKE', variable + '~%');
        }
      }).groupBy('date').then(rows => rows.map(row => ({ ...row,
        count: parseInt(row.count)
      })));
      return { ...graph,
        results: rows
      };
    },

    async countUniq(graph, from, to) {
      const variable = _lodash.default.first(graph.variables);

      const countUniq = await countUniqRecords(from, to, variable);
      const results = await this.count(graph, from, to);
      return { ...graph,
        ...results,
        countUniq
      };
    },

    percent: async function (graph, from, to) {
      const variable1 = _lodash.default.first(graph.variables);

      const variable2 = _lodash.default.last(graph.variables);

      const count1 = await getters.count({
        variables: [variable1]
      }, from, to);
      const count2 = await getters.count({
        variables: [variable2]
      }, from, to);

      const allDates = _lodash.default.uniq([..._lodash.default.map(count1.results, 'date'), ..._lodash.default.map(count2.results, 'date')]);

      const records = allDates.map(date => [_lodash.default.find(count1.results, {
        date
      }) || {
        count: 0,
        date
      }, _lodash.default.find(count2.results, {
        date
      }) || {
        count: 1,
        date
      }]);
      const results = records.map(([n1, n2]) => {
        const percent = _lodash.default.isFunction(graph.fn) ? graph.fn(_lodash.default.get(n1, 'count'), _lodash.default.get(n2, 'count')) : _lodash.default.get(n1, 'count') / _lodash.default.get(n2, 'count');
        return {
          date: n1.date,
          percent: percent > 1 ? 1 : percent
        };
      });
      let percent = undefined;

      if (graph.fnAvg) {
        const n1Uniq = await countUniqRecords(from, to, variable1);
        const n2Uniq = await countUniqRecords(from, to, variable2);
        percent = graph.fnAvg(n1Uniq, n2Uniq) * 100;
      }

      return { ...graph,
        results,
        percent
      };
    },
    piechart: async function (graph, from, to) {
      const variable = _lodash.default.first(graph.variables);

      const rows = await knex('analytics_custom').select(['name', knex.raw('sum(count) as count')]).where('date', '>=', from).andWhere('botId', botId).andWhere('date', '<=', to).andWhere('name', 'LIKE', variable + '~%').groupBy('name').then(rows => {
        return rows.map(row => {
          const name = _lodash.default.drop(row.name.split('~')).join('~');

          return { ...row,
            name: _lodash.default.isEmpty(name) ? 'unknown' : name,
            count: parseInt(row.count)
          };
        });
      });
      return { ...graph,
        results: rows
      };
    }
  };

  function getAll(from, to) {
    return Promise.map(graphs, graph => getters[graph['type']](graph, from, to));
  }

  return {
    increment,
    decrement,
    set,
    addGraph,
    getAll
  };
};

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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