"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = _interopRequireWildcard(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

var _mlKmeans = _interopRequireDefault(require("ml-kmeans"));

var _tmp = _interopRequireDefault(require("tmp"));

var _typings = require("../../typings");

var _preProcessor = require("./pre-processor");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = DEBUG('nlu').sub('slots');
const debugTrain = debug.sub('train');
const debugExtract = debug.sub('extract');
const debugVectorize = debug.sub('vectorize'); // TODO grid search / optimization for those hyperparams

const K_CLUSTERS = 15;
const KMEANS_OPTIONS = {
  iterations: 250,
  initialization: 'random',
  seed: 666 // so training is consistent

};
const CRF_TRAINER_PARAMS = {
  c1: '0.0001',
  c2: '0.01',
  max_iterations: '500',
  'feature.possible_transitions': '1',
  'feature.possible_states': '1'
};

class CRFExtractor {
  constructor(toolkit) {
    this.toolkit = toolkit;

    _defineProperty(this, "_isTrained", false);

    _defineProperty(this, "_ftModelFn", '');

    _defineProperty(this, "_crfModelFn", '');

    _defineProperty(this, "_ft", void 0);

    _defineProperty(this, "_tagger", void 0);

    _defineProperty(this, "_kmeansModel", void 0);
  }

  async load(traingingSet, languageModelBuf, crf) {
    // load language model
    const ftModelFn = _tmp.default.tmpNameSync({
      postfix: '.bin'
    });

    _fs.default.writeFileSync(ftModelFn, languageModelBuf);

    const ft = new this.toolkit.FastText.Model();
    await ft.loadFromFile(ftModelFn);
    this._ft = ft;
    this._ftModelFn = ftModelFn; // load kmeans (retrain because there is no simple way to store it)

    await this._trainKmeans(traingingSet); // load crf model

    this._crfModelFn = _tmp.default.tmpNameSync();

    _fs.default.writeFileSync(this._crfModelFn, crf);

    this._tagger = this.toolkit.CRF.createTagger();
    await this._tagger.open(this._crfModelFn);
    this._isTrained = true;
  }

  async train(trainingSet) {
    this._isTrained = false;

    if (trainingSet.length >= 2) {
      debugTrain('start training');
      debugTrain('training language model');
      await this._trainLanguageModel(trainingSet);
      debugTrain('training kmeans');
      await this._trainKmeans(trainingSet);
      debugTrain('training CRF');
      await this._trainCrf(trainingSet);
      debugTrain('reading tagger');
      this._tagger = this.toolkit.CRF.createTagger();
      await this._tagger.open(this._crfModelFn);
      this._isTrained = true;
      debugTrain('done training');
      return {
        language: (0, _fs.readFileSync)(this._ftModelFn),
        crf: (0, _fs.readFileSync)(this._crfModelFn)
      };
    } else {
      debugTrain('training set too small, skipping training');
      return {
        language: undefined,
        crf: undefined
      };
    }
  }
  /**
   * Returns an object with extracted slots name as keys.
   * Each slots under each keys can either be a single Slot object or Array<Slot>
   * return value example:
   * slots: {
   *   artist: {
   *     name: "artist",
   *     value: "Kanye West",
   *     entity: [Object] // corresponding sdk.NLU.Entity
   *   },
   *   songs : [ multiple slots objects here]
   * }
   */


  async extract(text, intentDef, entities) {
    debugExtract(text, {
      entities
    });
    const seq = (0, _preProcessor.generatePredictionSequence)(text, intentDef.name, entities);
    const tags = await this._tag(seq); // notice usage of zip here, we want to loop on tokens and tags at the same index

    return _lodash.default.zip(seq.tokens, tags).filter(([token, tag]) => {
      if (!token || !tag || tag === _typings.BIO.OUT) {
        return false;
      }

      const slotName = tag.slice(2);
      return intentDef.slots.find(slotDef => slotDef.name === slotName) !== undefined;
    }).reduce((slotCollection, [token, tag]) => {
      const slotName = tag.slice(2);

      const slot = this._makeSlot(slotName, token, intentDef.slots, entities);

      if (tag[0] === _typings.BIO.INSIDE && slotCollection[slotName]) {
        // prevent cases where entity has multiple tokens e.g. "4 months months"
        if (!slot.entity || token.end > slot.entity.meta.end) {
          // simply append the source if the tag is inside a slot
          slotCollection[slotName].source += ` ${token.value}`;
        }
      } else if (tag[0] === _typings.BIO.BEGINNING && slotCollection[slotName]) {
        // if the tag is beginning and the slot already exists, we create need a array slot
        if (Array.isArray(slotCollection[slotName])) {
          slotCollection[slotName].push(slot);
        } else {
          // if no slots exist we assign a slot to the slot key
          slotCollection[slotName] = [slotCollection[slotName], slot];
        }
      } else {
        slotCollection[slotName] = slot;
      }

      return slotCollection;
    }, {});
  } // this is made "protected" to facilitate model validation


  async _tag(seq) {
    if (!this._isTrained) {
      throw new Error('Model not trained, please call train() before');
    }

    const inputVectors = [];

    for (let i = 0; i < seq.tokens.length; i++) {
      const featureVec = await this._vectorize(seq.tokens, seq.intent, i);
      inputVectors.push(featureVec);
    }

    return this._tagger.tag(inputVectors).result;
  }

  _makeSlot(slotName, token, slotDefinitions, entities) {
    const slotDef = slotDefinitions.find(slotDef => slotDef.name === slotName);
    const entity = slotDef && entities.find(e => slotDef.entities.indexOf(e.name) !== -1 && e.meta.start <= token.start && e.meta.end >= token.end);

    const value = _lodash.default.get(entity, 'data.value', token.value);

    const source = _lodash.default.get(entity, 'meta.source', token.value);

    const slot = {
      name: slotName,
      value,
      source
    };

    if (entity) {
      slot.entity = entity;
    }

    return slot;
  }

  async _trainKmeans(sequences) {
    const tokens = _lodash.default.flatMap(sequences, s => s.tokens);

    const data = await Promise.mapSeries(tokens, t => this._ft.queryWordVectors(t.value));
    const k = data.length > K_CLUSTERS ? K_CLUSTERS : 2;

    try {
      this._kmeansModel = (0, _mlKmeans.default)(data, k, KMEANS_OPTIONS);
    } catch (error) {
      throw Error('Error training K-means model');
    }
  }

  async _trainCrf(sequences) {
    this._crfModelFn = _tmp.default.fileSync({
      postfix: '.bin'
    }).name;
    const trainer = this.toolkit.CRF.createTrainer();
    trainer.set_params(CRF_TRAINER_PARAMS);
    trainer.set_callback(str => {
      /* swallow training results */
    });

    for (const seq of sequences) {
      const inputVectors = [];
      const labels = [];

      for (let i = 0; i < seq.tokens.length; i++) {
        const featureVec = await this._vectorize(seq.tokens, seq.intent, i);
        inputVectors.push(featureVec);
        const labelSlot = seq.tokens[i].slot ? `-${seq.tokens[i].slot}` : '';
        labels.push(`${seq.tokens[i].tag}${labelSlot}`);
      }

      trainer.append(inputVectors, labels);
    }

    trainer.train(this._crfModelFn);
  }

  async _trainLanguageModel(samples) {
    this._ftModelFn = _tmp.default.fileSync({
      postfix: '.bin'
    }).name;

    const ftTrainFn = _tmp.default.fileSync({
      postfix: '.txt'
    }).name;

    const ft = new this.toolkit.FastText.Model();
    const trainContent = samples.reduce((corpus, seq) => {
      const cannonicSentence = seq.tokens.map(s => {
        if (s.tag === _typings.BIO.OUT) return s.value;else return s.slot;
      }).join(' ');
      return `${corpus}${cannonicSentence}\n`;
    }, '');

    _fs.default.writeFileSync(ftTrainFn, trainContent, 'utf8');

    const skipgramParams = {
      input: ftTrainFn,
      minCount: 2,
      dim: 15,
      lr: 0.05,
      epoch: 50,
      wordNgrams: 3
    };
    debugTrain('training skipgram', skipgramParams);
    await ft.trainToFile('skipgram', this._ftModelFn, skipgramParams);
    this._ft = ft;
  }

  async _vectorizeToken(token, intentName, featPrefix, includeCluster) {
    const vector = [`${featPrefix}intent=${intentName}`];
    if (token.value === token.value.toLowerCase()) vector.push(`${featPrefix}low`);
    if (token.value === token.value.toUpperCase()) vector.push(`${featPrefix}up`);
    if (token.value.length > 1 && token.value[0] === token.value[0].toUpperCase() && token.value[1] === token.value[1].toLowerCase()) vector.push(`${featPrefix}title`);

    if (includeCluster) {
      const cluster = await this._getWordCluster(token.value);
      vector.push(`${featPrefix}cluster=${cluster.toString()}`);
    }

    const entitiesFeatures = (token.matchedEntities.length ? token.matchedEntities : ['none']).map(ent => `${featPrefix}entity=${ent === 'any' ? 'none' : ent}`);
    return [...vector, ...entitiesFeatures];
  } // TODO maybe use a slice instead of the whole token seq ?


  async _vectorize(tokens, intentName, idx) {
    const prev = idx === 0 ? ['w[0]bos'] : await this._vectorizeToken(tokens[idx - 1], intentName, 'w[-1]', true);
    const current = await this._vectorizeToken(tokens[idx], intentName, 'w[0]', false);
    const next = idx === tokens.length - 1 ? ['w[0]eos'] : await this._vectorizeToken(tokens[idx + 1], intentName, 'w[1]', true);
    debugVectorize(`"${tokens[idx].value}" (${idx})`, {
      prev,
      current,
      next
    });
    return [...prev, ...current, ...next];
  }

  async _getWordCluster(word) {
    const vector = await this._ft.queryWordVectors(word);
    return this._kmeansModel.nearest([vector])[0];
  }

}

exports.default = CRFExtractor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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