"use strict";

require("bluebird-global");

var _pattern_extractor = _interopRequireDefault(require("./pattern_extractor"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

describe('Custom entity extraction', () => {
  require('../../../../../../src/bp/import-rewire');

  const {
    default: computeLevenshteinDistance
  } = require('../../../../../../src/bp/ml/homebrew/levenshtein');

  const {
    default: computeJaroWinklerDistance
  } = require('../../../../../../src/bp/ml/homebrew/jaro-winkler');

  const Toolkit = {
    Strings: {
      computeLevenshteinDistance,
      computeJaroWinklerDistance
    }
  };
  test('Extract pattern entities', async () => {
    const pattern = 'lol';
    const entityDef = {
      id: '_',
      name: 'Fun',
      type: 'pattern',
      pattern
    };
    const userInput = 'lollolppplol hello haha';
    const extractor = new _pattern_extractor.default(Toolkit);
    const entities = await extractor.extractPatterns(userInput, [entityDef]);
    expect(entities.length).toEqual(3);
    expect(entities[0].name).toEqual(entityDef.name);
    expect(entities[0].meta.start).toEqual(0);
    expect(entities[0].meta.end).toEqual(3);
    expect(entities[0].data.value).toEqual(pattern);
    expect(entities[1].name).toEqual(entityDef.name);
    expect(entities[1].meta.start).toEqual(3);
    expect(entities[1].meta.end).toEqual(6);
    expect(entities[1].data.value).toEqual(pattern);
    expect(entities[2].name).toEqual(entityDef.name);
    expect(entities[2].meta.start).toEqual(9);
    expect(entities[2].meta.end).toEqual(12);
    expect(entities[2].data.value).toEqual(pattern);
  });
  test('Extract fuzzy list entities', async () => {
    const entityDef = {
      id: '_',
      name: 'Cars',
      type: 'list',
      fuzzy: true,
      occurences: [{
        name: 'Mercedes-Benz',
        synonyms: ['Benz', 'Mercedez Benz', 'Mercedez', 'Merc']
      }, {
        name: 'BMW',
        synonyms: ['bm']
      }]
    };
    const userInput = `
I'm riding my mercedes-benz to the dealership then I will take my BM to buy an other mercedes because we need merchandise for the shop BMW!`;
    /*
    [============]                                      ==                 [======]                                          [=]
    */

    const extractor = new _pattern_extractor.default(Toolkit);
    const entities = await extractor.extractLists(userInput, 'en', [entityDef]);
    expect(entities.length).toEqual(4);
    expect(entities[0].name).toEqual(entityDef.name);
    expect(entities[0].meta.start).toEqual(15);
    expect(entities[0].meta.end).toEqual(28);
    expect(entities[0].meta.source).toEqual('mercedes-benz');
    expect(entities[0].data.value).toEqual('Mercedes-Benz');
    expect(entities[1].name).toEqual(entityDef.name);
    expect(entities[1].meta.start).toEqual(86);
    expect(entities[1].meta.end).toEqual(94);
    expect(entities[1].meta.source).toEqual('mercedes');
    expect(entities[1].data.value).toEqual('Mercedes-Benz');
    expect(entities[2].name).toEqual(entityDef.name);
    expect(entities[2].meta.start).toEqual(67);
    expect(entities[2].meta.end).toEqual(69);
    expect(entities[2].meta.source).toEqual('BM');
    expect(entities[2].data.value).toEqual('BMW');
    expect(entities[3].name).toEqual(entityDef.name);
    expect(entities[3].meta.start).toEqual(136);
    expect(entities[3].meta.end).toEqual(140);
    expect(entities[3].meta.source).toEqual('BMW!');
    expect(entities[3].data.value).toEqual('BMW');
  });
  test('Extract exact list entities', async () => {
    const entityDef = {
      id: '_',
      name: 'Artists',
      type: 'list',
      fuzzy: false,
      occurences: [{
        name: 'Kanye West',
        synonyms: ['Ye']
      }]
    };
    const userInput = `
My name is kanye West and I rap like kanye wsest`;
    /*
    [========]                [xxxxxxxxxx]
    */

    const extractor = new _pattern_extractor.default(Toolkit);
    const entities = await extractor.extractLists(userInput, 'en', [entityDef]);
    expect(entities.length).toEqual(1);
    expect(entities[0].name).toEqual(entityDef.name);
    expect(entities[0].meta.start).toEqual(12);
    expect(entities[0].meta.end).toEqual(22);
    expect(entities[0].meta.source).toEqual('kanye West');
    expect(entities[0].data.value).toEqual('Kanye West');
  });
  test('Extract fuzzy entities with synonyms not treated as fuzzy', async () => {
    const entityDef = {
      id: '_',
      name: 'People',
      type: 'list',
      fuzzy: true,
      occurences: [{
        name: 'Jon Gore',
        synonyms: ['Jon', 'Gore']
      }]
    };
    const userInput = `
    I can't hear about Jone Goree anymore. Jone is a clown, so is gore`;
    /*
    [========]          [xx]                   [==]
    */

    const extractor = new _pattern_extractor.default(Toolkit);
    const entities = await extractor.extractLists(userInput, 'en', [entityDef]);
    expect(entities.length).toEqual(2);
    expect(entities[0].name).toEqual(entityDef.name);
    expect(entities[0].meta.start).toEqual(67);
    expect(entities[0].meta.end).toEqual(71);
    expect(entities[0].meta.source).toEqual('gore');
    expect(entities[0].data.value).toEqual('Jon Gore');
    expect(entities[1].name).toEqual(entityDef.name);
    expect(entities[1].meta.start).toEqual(24);
    expect(entities[1].meta.end).toEqual(34);
    expect(entities[1].meta.source).toEqual('Jone Goree');
    expect(entities[1].data.value).toEqual('Jon Gore');
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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