"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _joi = require("joi");

var _ms = _interopRequireDefault(require("ms"));

var _confusionEngine = _interopRequireDefault(require("./confusion-engine"));

var _validation = require("./validation");

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const SYNC_INTERVAL_MS = (0, _ms.default)('15s');

var _default = async (bp, nlus) => {
  const router = bp.http.createRouterForBot('nlu');
  const syncByBots = {};

  const scheduleSyncNLU = botId => {
    if (syncByBots[botId]) {
      clearTimeout(syncByBots[botId]);
      delete syncByBots[botId];
    }

    syncByBots[botId] = setTimeout(() => {
      delete syncByBots[botId];
      const botEngine = nlus[botId];
      syncNLU(botEngine, false);
    }, SYNC_INTERVAL_MS);
  };

  const syncNLU = async (botEngine, confusionMode = false) => {
    const startTraining = {
      type: 'nlu',
      name: 'train',
      working: true,
      message: 'Training model'
    };
    bp.realtime.sendPayload(bp.RealTimePayload.forAdmins('statusbar.event', startTraining));

    if (confusionMode && botEngine instanceof _confusionEngine.default) {
      botEngine.computeConfusionOnTrain = true;
    }

    try {
      return await botEngine.sync(confusionMode);
    } catch (e) {
      bp.realtime.sendPayload(bp.RealTimePayload.forAdmins('toast.nlu-sync', {
        text: `NLU Sync Error: ${e.name} : ${e.message}`,
        type: 'error'
      }));
    } finally {
      if (confusionMode && botEngine instanceof _confusionEngine.default) {
        botEngine.computeConfusionOnTrain = false;
      }

      const trainingComplete = {
        type: 'nlu',
        name: 'done',
        working: false,
        message: 'Model is up-to-date'
      };
      bp.realtime.sendPayload(bp.RealTimePayload.forAdmins('statusbar.event', trainingComplete));
    }
  };

  router.get('/confusion/:modelHash', async (req, res) => {
    try {
      const matrix = await nlus[req.params.botId].storage.getConfusionMatrix(req.params.modelHash);
      res.send(matrix);
    } catch (err) {
      res.sendStatus(401);
    }
  });
  router.post('/confusion', async (req, res) => {
    try {
      const botEngine = nlus[req.params.botId];
      const modelHash = await syncNLU(botEngine, true);
      res.send({
        modelHash
      });
    } catch (err) {
      res.status(400).send('Could not train confusion matrix');
    }
  });
  router.get('/intents', async (req, res) => {
    res.send((await nlus[req.params.botId].storage.getIntents()));
  });
  router.get('/intents/:intent', async (req, res) => {
    res.send((await nlus[req.params.botId].storage.getIntent(req.params.intent)));
  });
  router.delete('/intents/:intent', async (req, res) => {
    const botEngine = nlus[req.params.botId];
    await botEngine.storage.deleteIntent(req.params.intent);
    scheduleSyncNLU(req.params.botId);
    res.sendStatus(204);
  });
  router.post('/intents', async (req, res) => {
    try {
      const intentDef = await (0, _joi.validate)(req.body, _validation.IntentDefCreateSchema, {
        stripUnknown: true
      });
      const botEngine = nlus[req.params.botId];
      await botEngine.storage.saveIntent(intentDef.name, intentDef);
      scheduleSyncNLU(req.params.botId);
      res.sendStatus(201);
    } catch (err) {
      bp.logger.attachError(err).warn('Cannot create intent, invalid schema');
      res.status(400).send('Invalid schema');
    }
  });
  router.get('/contexts', async (req, res) => {
    const botId = req.params.botId;
    const filepaths = await bp.ghost.forBot(botId).directoryListing('/intents', '*.json');
    const contextsArray = await Promise.map(filepaths, async filepath => {
      const file = await bp.ghost.forBot(botId).readFileAsObject('/intents', filepath);
      return file['contexts'];
    }); // Contexts is an array of arrays that can contain duplicate values

    res.send(_lodash.default.uniq(_lodash.default.flatten(contextsArray)));
  });
  router.get('/entities', async (req, res) => {
    const entities = await nlus[req.params.botId].storage.getAvailableEntities();
    res.json(entities);
  });
  router.post('/entities', async (req, res) => {
    const {
      botId
    } = req.params;

    try {
      const entityDef = await (0, _joi.validate)(req.body, _validation.EntityDefCreateSchema, {
        stripUnknown: true
      });
      const botEngine = nlus[botId];
      await botEngine.storage.saveEntity(entityDef);
      scheduleSyncNLU(req.params.botId);
      res.sendStatus(201);
    } catch (err) {
      bp.logger.attachError(err).warn('Cannot create entity, imvalid schema');
      res.status(400).send('Invalid schema');
    }
  });
  router.put('/entities/:id', async (req, res) => {
    const content = req.body;
    const {
      botId,
      id
    } = req.params;
    const updatedEntity = content;
    const botEngine = nlus[botId];
    await botEngine.storage.saveEntity({ ...updatedEntity,
      id
    });
    scheduleSyncNLU(req.params.botId);
    res.sendStatus(201);
  });
  router.delete('/entities/:id', async (req, res) => {
    const {
      botId,
      id
    } = req.params;
    const botEngine = nlus[botId];
    await botEngine.storage.deleteEntity(id);
    scheduleSyncNLU(req.params.botId);
    res.sendStatus(204);
  });
  router.post('/extract', async (req, res) => {
    const eventText = {
      preview: req.body.text,
      payload: {
        text: req.body.text
      }
    };

    try {
      const result = await nlus[req.params.botId].extract(eventText.preview);
      res.send(result);
    } catch (err) {
      res.status(500).send(`Error extracting NLU data from event: ${err}`);
    }
  });
};

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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