#!/bin/sh

# Copyright (C) 2011 Help/Systems LLC, All Rights Reserved
# Basic install script for Automate Schedule Agent when the 
# advanced Perl installer is unusable.

umask 066

PATH=$PATH:/usr/ucb

WHOAMI=`whoami`
DEFAULT_DIR="/opt/automate-schedule/agent"

JAVA_EXEC=""
JAR_EXEC=""
BAD_JAVA=""
# directory to which the product will be installed
INSTALL_DIR=""

START_DIR=`pwd`
FILES_DIR="files-agent"
. $FILES_DIR/buildinfo.txt
ECHO_PRE_NO_NEWLINE=""
ECHO_POST_NO_NEWLINE=""
OS=""
RUN_ADMIN_CONFIG=0
EXISTING_PORT_NUMBER=""
PORT_NUMBER=""
CLASSPATH=""
DEFAULT_PORT="7471"
SERVER_ADDRESS=""
SERVER_PORT=""
CONFIG_FILE="agent.xml"
RAW_OP_SYS=`uname`
PROCESSOR=""
JAVA_FLAGS=""
BIT_WIDTH="32"

# netstat command is deprecated on linux, and we should use ss instead
# I need to check if ss exists, however, because AIX still uses netstat
PORTSTAT_CMD='ss -an'
command -v ss >/dev/null 2>&1
if [ $? -eq 1 ]; then
    PORTSTAT_CMD='netstat -an'
fi    

if [ "${RAW_OP_SYS}" = "Linux" -o "${RAW_OP_SYS}" = "HP-UX" ]; then
    PROCESSOR=`uname -m`
else
    PROCESSOR=`uname -p`
fi

if [ "${RAW_OP_SYS}" = "HP-UX" ]; then
    has_slash=`echo $PROCESSOR | grep / | wc -l`
    if [ $has_slash -eq 1 ]; then
        PROCESSOR=`echo $PROCESSOR | cut -f1 -d/`
    fi
fi

if [ "${RAW_OP_SYS}" = "AIX" -a "${PROCESSOR}" = "powerpc" ]; then
    BIT_WIDTH=`prtconf -k | cut -f2 -d: | cut -f1 -d- | sed s/\ //g`
    if [ "64" = "${BIT_WIDTH}" ]; then
        PROCESSOR=${PROCESSOR}$BIT_WIDTH
    fi
fi


if [ "${RAW_OP_SYS}" = "SunOS" -a "${PROCESSOR}" = "sparc" ]; then
    BIT_WIDTH=`isainfo -b`
    if [ "64" = "${BIT_WIDTH}" ]; then
        PROCESSOR=${PROCESSOR}$BIT_WIDTH
        JAVA_FLAGS="-d${BIT_WIDTH}"
    fi
fi
 
if [ "${RAW_OP_SYS}" = "Darwin" ]; then
    # make this work just like installAgent.pl/Install.pm
    PROCESSOR="x86_64"
    BIT_WIDTH="64"
fi

PLATFORM=${RAW_OP_SYS}/${PROCESSOR}    

checkEcho() {
    linux=`uname -a | grep -i linux | wc -l`
    solaris=`uname -a | grep -i sunos | wc -l`
    aix=`uname -a | grep -i aix | wc -l`
    hp=`uname -a | grep -i HP-UX | wc -l`

    if [ 0 -ne "${linux}" ]; then
        OS="Linux"
        ECHO_PRE_NO_NEWLINE="-n "
    fi
    if [ 0 -ne "${aix}" ]; then
        OS="AIX"
        ECHO_POST_NO_NEWLINE="\\c"
    fi
    if [ 0 -ne "${solaris}" ]; then
        OS="Solaris"
        ECHO_POST_NO_NEWLINE="\\c"
    fi
    if [ 0 -ne "${hp}" ]; then
        OS="HP-UX"
        ECHO_POST_NO_NEWLINE="\\c"
    fi
}

checkJarCommand() {
    JAR_EXEC="${JAVA_HOME}/bin/java sun.tools.jar.Main"
}

checkJavaHome() {
    if [ -z "${JAVA_HOME}" ]; then
        echo "This system does not have Java installed, or the JAVA_HOME"
        echo "   environment variable does not point to it."
        BAD_JAVA="y"
    else
        echo Checking Java in ${JAVA_HOME}
        JAVA_EXEC="${JAVA_HOME}/bin/java"
    fi
}

checkJavaVersion() {
    if [ -x "${JAVA_EXEC}" ]; then
        ${JAVA_EXEC} ${JAVA_FLAGS} -version > /tmp/javaVersion.txt 2>&1
        java12=`head -1 /tmp/javaVersion.txt | grep "\"1\.2" | wc -l`
        java13=`head -1 /tmp/javaVersion.txt | grep "\"1\.3" | wc -l`
        java14=`head -1 /tmp/javaVersion.txt | grep "\"1\.4" | wc -l`
        java15=`head -1 /tmp/javaVersion.txt | grep "\"1\.5" | wc -l`
        java16=`head -1 /tmp/javaVersion.txt | grep "\"1\.6" | wc -l`
        java17=`head -1 /tmp/javaVersion.txt | grep "\"1\.7" | wc -l`
        rm -f /tmp/javaVersion.txt
    else
        BAD_JAVA="y"
    fi

    if [ -z "${java12}" ]; then
        java12=0
    fi
    if [ 0 -ne "${java12}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.2, which is not usable."
        BAD_JAVA="y"
        JAVA_EXEC=""
    fi
    
    if [ -z "${java13}" ]; then
        java13=0
    fi
    if [ 0 -ne "${java13}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.3, which is not usable."
        BAD_JAVA="y"
        JAVA_EXEC=""
    fi
    
    if [ -z "${java14}" ]; then
        java14=0
    fi
    if [ 0 -ne "${java14}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.4, which is not usable."
        BAD_JAVA="y"
        JAVA_EXEC=""
    fi
    
    if [ -z "${java15}" ]; then
        java15=0
    fi
    if [ 0 -ne "${java15}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.5, which is not usable."
        BAD_JAVA="y"
        JAVA_EXEC=""
    fi

    if [ -z "${java16}" ]; then
        java16=0
    fi
    if [ 0 -ne "${java16}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.6."
    fi
    
    if [ -z "${java17}" ]; then
        java17=0
    fi
    if [ 0 -ne "${java17}" ]; then
        echo "Your JAVA_HOME points to ${JAVA_HOME}, which appears to be version 1.7."
    fi

    if [ -x "${JAVA_EXEC}" ]; then
        ${JAVA_EXEC} ${JAVA_FLAGS} -cp ${FILES_DIR}/javatests.jar com.helpsystems.enterprise.install.JVMTests -a
        if [ $? -ne 0 ]; then
            echo The Java installation at ${JAVA_EXEC} is not usable. This product needs an Oracle-certified Java installation.
            BAD_JAVA="y";
        fi
    fi
}

buildClassPath() {
    CLASSPATH=""
    lib_dir=${FILES_DIR}/lib
    liblist=`ls ${lib_dir}`
    for i in ${liblist}; do
        CLASSPATH=${CLASSPATH}":"$lib_dir"/"$i
    done
}


checkIfConfigNeeded() {
    # see if stuff has already been configured.
    config_file=${INSTALL_DIR}/${CONFIG_FILE}
    
    if [ -f "${config_file}" ]; then
        EXISTING_PORT_NUMBER=`grep ListenPort ${config_file} | cut -f6 -d'"'`
    fi
    if [ -z "${EXISTING_PORT_NUMBER}" ]; then
        # the port isn't set. we have to run the config stuff
        RUN_ADMIN_CONFIG=1
    fi

    if [ "${RUN_ADMIN_CONFIG}" -eq 1 ]; then
        return;
    fi
    
    # the configuration already has an admin user defined.
    echo "A configuration file already exists."
    echo ${ECHO_PRE_NO_NEWLINE} "Would you like to keep the current configuration? [y] "${ECHO_POST_NO_NEWLINE} 
    while [ 1 -lt 2 ]; do
        read confirm
        if [ -z "${confirm}" -o "y" = "${confirm}" -o "Y" = "${confirm}" ]; then
            RUN_ADMIN_CONFIG=0
            break
        elif [ "n" = "${confirm}" -o "N" = "${confirm}" ]; then
            rm -f ${config_file}
            RUN_ADMIN_CONFIG=1
            break
        else 
            echo ${ECHO_PRE_NO_NEWLINE} "Please answer Y or N: [y] "${ECHO_POST_NO_NEWLINE} 
        fi
    done
}

configInfo() {
    cd ${INSTALL_DIR}
    ./config.sh -f ${CONFIG_FILE} -lp ${PORT_NUMBER} -sa ${SERVER_ADDRESS} -sp ${SERVER_PORT}
    chmod 644 ${CONFIG_FILE}
    cd ${START_DIR}
}


configScripts() {
    # Note to self: we can't use / as the delimiter in the sed command, because
    # the stuff we're substituting in contains those same slashes. Thus, we'll
    # get sneaky and use question marks.
    cat ${INSTALL_DIR}/runjava.sh | sed "s?XXX?${JAVA_HOME}?g"  > /tmp/runjava.sh
    mv -f /tmp/runjava.sh ${INSTALL_DIR}/runjava.sh
    chmod 755 ${INSTALL_DIR}/runjava.sh

}

moveFiles() {

    echo "Copying product files..."

    if [ ! -d ${INSTALL_DIR}/logs ]; then
        mkdir ${INSTALL_DIR}/logs
        if [ $? -ne 0 ]; then
            echo "Unable to create directory ${INSTALL_DIR}/logs"
            exit 1
        fi
    fi

    if [ ! -d ${INSTALL_DIR}/joblogtmp ]; then
        mkdir ${INSTALL_DIR}/joblogtmp
        if [ $? -ne 0 ]; then
            echo "Unable to create directory ${INSTALL_DIR}/joblogtmp"
            exit 1
        fi
        chmod 700 ${INSTALL_DIR}/joblogtmp
        if [ $? -ne 0 ]; then
            echo "Warning: Unable to change the permissions on directory ${INSTALL_DIR}/joblogtmp"
        fi
    fi
    
    if [ "$OS" = "AIX" ]; then
        LIBNSZDIR=${INSTALL_DIR}/bin/${PLATFORM}
        if [ -d $LIBNSZDIR ]; then
            # Bug 13913: flush loaded shared libraries on AIX
            #   or not - /usr/sbin/slibclean can only be run as root.
            if [ `whoami` = "root" ]; then
                /usr/sbin/slibclean
            fi
            
            # Otherwise, a "well known" AIX "workaround" is to just rename it.
            rm -rf ${INSTALL_DIR}/bin/trash > /dev/null 2>&1
            mv -f $LIBNSZDIR ${INSTALL_DIR}/bin/trash
            if [ $? -ne 0 ]; then
                echo "Unable to remove $LIBNSZDIR"
                exit 1
            fi

        fi
    fi

    if [ -d "${INSTALL_DIR}/lib/sigarDlls" ]; then
        rm -rf ${INSTALL_DIR}/lib/trash
        mv -f ${INSTALL_DIR}/lib/sigarDlls ${INSTALL_DIR}/lib/trash
        if [ $? -gt 0 ]; then
            echo "Unable to remove ${INSTALL_DIR}/lib/sigarDlls"
            exit 1
        fi
    fi
    
    existingNopwd=${INSTALL_DIR}/bin/${PLATFORM}/nopwd
    if [ -f $existingNopwd -a ! -w $existingNopwd ]; then
        existingCksum=`sum -r $existingNopwd | cut -f1 -d' '`
        if [ $? -ne 0 ]; then
            echo "Unable to obtain the checksum of $existingNopwd"
            exit 1
        fi

        newNopwd=${FILES_DIR}/bin/${PLATFORM}/nopwd
        if [ -f $newNopwd ]; then
            newCksum=`sum -r $newNopwd | cut -f1 -d' '`
            if [ $? -ne 0 ]; then
                echo "Unable to obtain the checksum of $newNopwd"
                exit 1
            fi
        
            if [ $existingCksum -eq $newCksum ]; then
                # delete the new one, since it's the same as the existing one,
                #    and we won't be able to overwrite the existing one.
                rm -f $newNopwd
                if [ $? -ne 0 ]; then 
                    echo "Unable to remove $newNopwd"
                    exit 1
                fi
            else
                echo "Unable to overwrite $existingNopwd, you may need 'root' to remove the file or run this installer."
                exit 1
            fi
        fi
    fi

    cp -R -p ${FILES_DIR}/* ${INSTALL_DIR}
    if [ $? -ne 0 ]; then
        echo "There was a problem copying over the required files."
        exit 1
    fi
    chmod 755 ${INSTALL_DIR}/*.sh
    chmod 755 ${INSTALL_DIR}/lib
    chmod 755 ${INSTALL_DIR}/lib/*
    chmod 755 ${INSTALL_DIR}/r
    chmod 644 ${INSTALL_DIR}/r/*

    if [ "$OS" = "AIX" ]; then
        # move the SAP library back, if necessary.
        trashedSAPlib=${INSTALL_DIR}/bin/trash/libsapjco3.so
        normalSAPlib=${INSTALL_DIR}/bin/${PLATFORM}
        if [ -f $trashedSAPlib ]; then
            cp $trashedSAPlib $normalSAPlib
        fi
    fi
    
}



getInstallDir() {
    while [ -z "${INSTALL_DIR}" ]; do
        echo  ${ECHO_PRE_NO_NEWLINE} "Where would you like to install the agent? [${DEFAULT_DIR}] "${ECHO_POST_NO_NEWLINE} 
        read dirname
        if [ -z "${dirname}" ]; then
            dirname=$DEFAULT_DIR
        fi

        if [ ! -x "${dirname}" ]; then
            mkdir -p ${dirname}
            if [ -x "${dirname}" ]; then
                chmod 755 ${dirname};
            fi
        fi

        if [ -x "${dirname}" ]; then
            cd ${dirname}
            INSTALL_DIR=`pwd`
            cd ${START_DIR}
		else
            echo "The specified directory ${dirname} could not be created. Please check your permissions."
			exit 1
        fi
    done
}

checkForRunningAgents() {

    pid_file_count=`ls $INSTALL_DIR/*.pid 2> /dev/null | wc -l`
    if [ $pid_file_count -eq 0 ]; then
        return;
    fi
    agents_running=0
    for file in `ls $INSTALL_DIR/*.pid`; do
        if [ -r $file ]; then
            OLD_PID=`cat ${file}`
            PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`
            if [ 0 -ne "${PROCESS_COUNT}" ]; then
                echo "Agent process $OLD_PID needs to be shutdown before installation can continue."
                agents_running=1
            fi
        fi
    done
    if [ $agents_running -eq 1 ]; then
        exit 1;
    fi
}

verifyPort() {

        expr $1 + 0 > /dev/null 2>&1
        VALID_PORT=$?

        if [ $VALID_PORT -ne 0 ]; then
            echo "Please enter a number between 1 and 65535";
            return 0
        fi

        if [ $1 -lt 1 ]; then
            echo "Please enter a number between 1 and 65535";
            return 0
        fi

        if [ $1 -gt 65535 ]; then
            echo "Please enter a number between 1 and 65535";
            return 0
        fi
        
        return 1
}    

getPortNumber() {

    while [ -z "${PORT_NUMBER}" ]; do
        echo ${ECHO_PRE_NO_NEWLINE} "Enter the port number to use for local connections [${DEFAULT_PORT}] "${ECHO_POST_NO_NEWLINE} 
        read PORT_NUMBER
        if [ -z "${PORT_NUMBER}" ]; then
            PORT_NUMBER=${DEFAULT_PORT}
        fi

        verifyPort "$PORT_NUMBER"
        VALID_PORT=$?
        if [ $VALID_PORT -eq 0 ]; then
            PORT_NUMBER=""
            continue
        fi

        
        in_use=`${PORTSTAT_CMD} | grep LISTEN | egrep "[^0-9]${PORT_NUMBER}" | wc -l`
        if [ 1 -eq "${in_use}" ]; then
            echo "The port ${PORT_NUMBER} is currently in use. Please exit the installer and shutdown the process that is using the port, or choose a different port."
            PORT_NUMBER=""
        fi
    done
}


getServerInfo() {

    while [ -z "${SERVER_ADDRESS}" ]; do
        echo ${ECHO_PRE_NO_NEWLINE} "Enter the name or IP address of the Automate Schedule Server: "${ECHO_POST_NO_NEWLINE} 
        read SERVER_ADDRESS
    done

    while [ -z "${SERVER_PORT}" ]; do
        echo ${ECHO_PRE_NO_NEWLINE} "Enter the agent-connection port of the Automate Schedule Server [7472] "${ECHO_POST_NO_NEWLINE} 
        read SERVER_PORT
        if [ -z "${SERVER_PORT}" ]; then
            SERVER_PORT=7472
        fi

        verifyPort "$SERVER_PORT"
        VALID_PORT=$?
        if [ $VALID_PORT -eq 0 ]; then
            SERVER_PORT=""
            continue
        fi
        
    done
}


buildLocalURL() {
    HOSTNAME=`hostname | cut -f1 -d'.'`
    LOCAL_URL="https://${HOSTNAME}:${PORT_NUMBER}"
}

prepareInstall() {

    if [ ! -d ${FILES_DIR}/bin/${PLATFORM} ]; then
        echo "The platform ${PLATFORM} is not supported - 'bin' dir not present."
        exit 1
    fi

    # see if we can resolve the libraries in nativesz
    nativesz=`ls ${FILES_DIR}/bin/${PLATFORM}/*native* 2> /dev/null`
    if [ -z "${nativesz}" ]; then
        echo "The platform ${PLATFORM} is not supported - native library is missing."
        exit 1
    fi

    chmod -R 755 ${FILES_DIR}/bin 2>&1 | grep -v nopwd

    LIB_DIR=${FILES_DIR}/lib/
    liblist=`ls ${LIB_DIR}*.jar`
    CLASSPATH=${FILES_DIR}/javatests.jar
    for i in ${liblist}; do
        CLASSPATH=${CLASSPATH}:${i}
    done

    # Run another java test to make sure the library is usable.
    ${JAVA_EXEC} ${JAVA_FLAGS} -Djava.library.path=${FILES_DIR}/bin/${PLATFORM} -cp ${CLASSPATH} com.helpsystems.enterprise.install.JVMTests -n
    result=$?
    if [ $result -ne 0 ]; then
        echo "Installation cannot continue due to failed library dependencies."
        exit 1
    fi

}


checkEcho

#echo "Welcome to Automate Schedule Agent, version ${automate_schedule_version} build ${automate_schedule_build_date}"
welcomeString=" Welcome to Automate Schedule Agent, version ${automate_schedule_version} build ${automate_schedule_build_date}";
currentDatef=`date`;
finalEchof=$currentDatef$welcomeString;
echo $finalEchof
echo "The current user is ${WHOAMI}, this system is ${PLATFORM}, ${BIT_WIDTH} bits"
echo ""

checkJavaHome
checkJavaVersion
checkJarCommand

if [ -n "${BAD_JAVA}" ]; then
    echo "Installation cannot continue until Java 1.6 or later"
    echo "   has been installed. If you already have Java installed, be sure that"
    echo "   the environment variable JAVA_HOME points to the directory."
    echo ""
    echo "For more information, please visit http://www.helpsystems.com/getjava"
    exit 1
fi

LD_LIBRARY_PATH=$LD_LIBRARY_PATH:${FILES_DIR}/bin/${PLATFORM}
export LD_LIBRARY_PATH

prepareInstall
buildClassPath

getInstallDir



checkForRunningAgents

checkIfConfigNeeded
if [ "${RUN_ADMIN_CONFIG}" -eq 1 ]; then
    getServerInfo
    getPortNumber
else
    # use the config we already have.
    PORT_NUMBER=${EXISTING_PORT_NUMBER}
    
    # make sure the thing is shutdown
    in_use=`${PORTSTAT_CMD}  | grep LISTEN | egrep "[^0-9]${PORT_NUMBER}" | wc -l`
    if [ 1 -eq "${in_use}" ]; then
        echo "The port ${PORT_NUMBER} is currently in use. Please shutdown the process that is using the port."
        exit 1
    fi
fi


buildLocalURL

moveFiles

configScripts

if [ "${RUN_ADMIN_CONFIG}" -eq 1 ]; then
    configInfo
fi

completionString="Installation completed at "
currentDate=`date "+%Y-%m-%d %H:%m:%S"`
finalEcho=$completionString$currentDate

echo $finalEcho

echo ${ECHO_PRE_NO_NEWLINE} "Would you like to start the Agent now? [y] "${ECHO_POST_NO_NEWLINE} 
while [ 1 -lt 2 ]; do
    read confirm
    if [ -z "${confirm}" -o "y" = "${confirm}" -o "Y" = "${confirm}" ]; then
        echo "You may want to add the command: ${INSTALL_DIR}/start.sh"
        echo "   to your server's startup sequence."
        echo "The process may take a minute or two to start. Use the link"
        echo "   ${LOCAL_URL} to access the diagnostic page."
        nohup ${INSTALL_DIR}/start.sh  >/dev/null 2>/dev/null &
        if [ $? != 0 ]; then
            echo "The agent failed to start, try manually running ${INSTALL_DIR}/start.sh"
        fi
        break
    elif [ "n" = "${confirm}" -o "N" = "${confirm}" ]; then
        echo "When you're ready to start the server, run the command: ${INSTALL_DIR}/start.sh"
        echo "You may want to add this command to your server's startup sequence."
        echo "After you've started the software, use the link"
        echo "   ${LOCAL_URL} to access the diagnostic page."
        break
    else
        echo ${ECHO_PRE_NO_NEWLINE} "Please answer Y or N: [y] "${ECHO_POST_NO_NEWLINE} 
    fi
done
