#!/bin/sh

# Copyright (C) 2010 Help/Systems LLC, All Rights Reserved
# Allows user to send a message to Alert, Network, Console, or Schedule

logDebug="false"
MAIN_CLASS="com.helpsystems.enterprise.peer.AgentControl"

# Initialize a verbose parameter
# "verbose=" to turn off verbose mode
# "verbose=-v" to turn on verbose mode
verbose=

# initialize empty parm flags
CFG_FILE=agent.xml
COMMAND=sendmsg
parm_mode=
parm_r=
parm_t=
parm_c=
parm_j=
parm_s=

# ------------------------------------------------------------------------
# Function: Log an error message                                         -
# ------------------------------------------------------------------------
logError() {
  printf "$1\n">&2
}

# ------------------------------------------------------------------------
# Function: Log an info message                                         -
# ------------------------------------------------------------------------
logInfo() {
  printf "$1\n"
}

# ------------------------------------------------------------------------
# Function: Log a debug message                                          -
# ------------------------------------------------------------------------
debugMsg() {
  if test "$logDebug" = "true"
  then 
    logInfo "DEBUG: $1"
  fi
}

# ------------------------------------------------------------------------
# Function: Show Usage Text                                              -
# ------------------------------------------------------------------------
showUsageText() {
  logInfo ""
  logInfo "Usage: $0 [-f config file]"
  logInfo "If no config file is provided, then the file agent.xml is used.

          -h show command help.
          -m <[EMAIL][TRAP]> 
          -r <[email addresses] when used with -m EMAIL
          -t <message text>
          -c <SNMP Trap severity code> AT, WR, or IN 
          [-s <email message subject>] Required with EMAIL
          [-v] verbose mode"
  logInfo ""
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  logInfo ""
  logInfo "Help for: $0"
  logInfo ""
  logInfo "SENDMSG - Allows a user to send a message via email or an SNMP Trap."
  showUsageText
  logInfo ""
  logInfo "  Sending an email message:"
  logInfo "    $0 [-f configfile] -m EMAIL -r <email addresses> "
  logInfo "    -t <message text> -s <subject>"
  logInfo ""
  logInfo "    Example: $0 -m EMAIL -r SOMEBODY@SOMEWHERE.COM,SOMEBODY2@SOMEWHERE.COM "
  logInfo "    -t \"I need to tell you something.\" -s \"The Subject\""
  logInfo ""
  logInfo ""
  logInfo "  Sending an SNMP Trap:"
  logInfo "    $0 [-f configfile] -m TRAP -c <SNMP Trap severity code>"
  logInfo "    -t <message text>"
  logInfo ""
  logInfo "    Example: $0 -m TRAP -c IN -t \"I need to tell you something.\""
  logInfo ""
  logInfo ""
}

# ------------------------------------------------------------------------
# Main Line Code.                                                        -
# ------------------------------------------------------------------------

if [ $# -eq 0 ]; then
  logError "Message parameters are required."
  showUsageText
  exit 1
fi

if [ "$1" = "--help" ]; then
  showHelpText
  exit 0
fi

#Check for config file first
if [ "$1" = "-f" ]; then
   if [ -z "$2" ]; then
     echo "Error: A configuration file must follow the -f parameter"
     exit 1
   fi
fi

while getopts :hvf:m:r:t:c:s: opt
  do
    case $opt in
    h)   showHelpText
         exit 0
         ;;
    v)   verbose=-v
         debugMsg "Setting on verbose mode"
         ;;
    f)   CFG_FILE=$OPTARG
         debugMsg "Config file: $CFG_FILE"
         ;;
    m)   parm_mode=$OPTARG
         debugMsg "parm_mode: $parm_mode"
         ;;
    r)   parm_r=$OPTARG
         debugMsg "parm_r: $parm_r"
         ;;
    t)   parm_t=$OPTARG
         debugMsg "parm_t: $parm_t"
         ;;
    c)   parm_c=$OPTARG
         debugMsg "parm_c: $parm_c"
         ;;
    s)   parm_s=$OPTARG
         debugMsg "parm_s: $parm_s"
         ;;
    '?') logError "$0: invalid option -$OPTARG"
         showUsageText
         exit 1
         ;;
    esac
  done

debugMsg "----------------------------------------------------------------"
debugMsg "- Verbose flag  . . . . . . : ${verbose}"
debugMsg "- Using config file . . . . : ${CFG_FILE}"
debugMsg "- Sending Command . . . . . : ${COMMAND}"
debugMsg "- Mode  . . . . . . . . . . : ${parm_mode}"
debugMsg "- R-Parameter . . . . . . . : ${parm_r}"
debugMsg "- T-Parameter . . . . . . . : ${parm_t}"
debugMsg "- C-Parameter . . . . . . . : ${parm_c}"
debugMsg "- S-Parameter . . . . . . . : ${parm_s}"
debugMsg "----------------------------------------------------------------"

PATH=$PATH:/usr/ucb

WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}

if [ ! -f ${CFG_FILE} ]; then
    logError "The config file ${CFG_FILE} does not exist. Command failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "The config file $CFG_FILE is not readable by the current user."
    exit 1
fi

# Parameter notes:
# - There are double quotes around the parms that could contain embedded blanks.
# - The verbose parameter is not quoted because it is optional.
echo "About to send a message using the Agent with configuration file $CFG_FILE ..."
./runjava.sh $MAIN_CLASS ${verbose} "${CFG_FILE}" ${COMMAND} -m "${parm_mode}" -r "${parm_r}" -t "${parm_t}" -s "${parm_s}" -c "${parm_c}"
