#!/bin/sh

# Copyright (C) 2017 HelpSystems, LLC, All Rights Reserved

logDebug="false"
MAIN_CLASS="com.helpsystems.enterprise.peer.AgentControl"

# Initialize a verbose parameter
#   - "VERBOSE=" to turn off verbose mode
#   - "VERBOSE=-v" to turn on verbose mode
VERBOSE=

# initialize empty parm flags
CFG_FILE=agent.xml
COMMAND=ROVAR
MODE=GET

# ------------------------------------------------------------------------
# Function: Log an error message                                         -
# ------------------------------------------------------------------------
logError() {
  printf "$1\n">&2
}

# ------------------------------------------------------------------------
# Function: Log an info message                                         -
# ------------------------------------------------------------------------
logInfo() {
  printf "$1\n"
}

# ------------------------------------------------------------------------
# Function: Log a debug message                                          -
# ------------------------------------------------------------------------
debugMsg() {
  if test "$logDebug" = "true"
  then 
    logInfo "DEBUG: $1"
  fi
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showUsageText() {
  logInfo ""
  logInfo "Usage: $0 [-h] [-f config file] [-v] <Automate Schedule variable>"
  logInfo "If no config file is provided, then the file agent.xml is used."
  logInfo ""
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  logInfo ""
  logInfo "Help for: $0"
  logInfo ""
  logInfo "GETSV - This command will retrieve the value of an Automate Schedule"
  logInfo "        Variable from the Automate Schedule server.  The value is"
  logInfo "        written to the standard output stream."
  showUsageText
  logInfo "  The -h option will show command help."
  logInfo "  The -v option will run in verbose logging mode."
  logInfo ""
  logInfo "  Environment variable names are case sensitive and will keep"
  logInfo "  the case in which they are entered."
  logInfo ""
  logInfo "  Examples:"
  logInfo "     $0 var1"
  logInfo "     $0 Var2"
  logInfo "     $0 -f agent2.xml var1"
  logInfo "     $0 -v VAR3"
  logInfo "     $0 -f agent2.xml -v VAR3"
  logInfo ""
  logInfo "  Exit Status:"
  logInfo "           0 = Success"
  logInfo "    non-zero = Failure"
  logInfo ""
  logInfo "Copyright (C) 2017 HelpSystems, LLC, All Rights Reserved"
  logInfo ""
}

# ------------------------------------------------------------------------
# Main Line Code.                                                        -
# ------------------------------------------------------------------------

if [ $# -gt 0 -a "$1" = "--help" ]; then
  showHelpText
  exit 0
fi

#Check for config file first
if [ "$1" = "-f" ]; then
   if [ -z "$2" ]; then
     echo "Error: A configuration file must follow the -f parameter"
     exit 1
   fi
fi

#-----------------------------------------------
# Process leading options.
# "Shift" them off the parm list.
#-----------------------------------------------
while getopts :vehf: opt
  do
    case $opt in
    f)   CFG_FILE=$OPTARG
         debugMsg "Using agent configuration file: $CFG_FILE"
         ;;
    h)   debugMsg "Processing -h (help) option..."
         showHelpText
         exit 0
         ;;
    v)   debugMsg "Setting on verbose mode..."
         VERBOSE=-v
         ;;
    '?') logError "$0: invalid option -$OPTARG"
         showUsageText
         exit 1
         ;;
    esac
  done

shift $((OPTIND-1))

#-----------------------------------------------
# One variable must be specified.
#-----------------------------------------------
if [ $# -eq 0 ]; then
  logError "An Automate Schedule Variable must be specified."
  showUsageText
  exit 1
fi

debugMsg "----------------------------------------------------------------"
debugMsg "- Verbose flag  . . . . . . : ${VERBOSE}"
debugMsg "- Using config file . . . . : ${CFG_FILE}"
debugMsg "- Sending Command . . . . . : ${COMMAND}"
debugMsg "- Mode  . . . . . . . . . . : ${MODE}"
debugMsg "----------------------------------------------------------------"
for i in "$@"
 do
debugMsg " Parameter  . . . . . . . . : $i"
 done
debugMsg "----------------------------------------------------------------"

PATH=$PATH:/usr/ucb

WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}

if [ ! -f ${CFG_FILE} ]; then
    logError "The config file ${CFG_FILE} does not exist. Command failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "The config file $CFG_FILE is not readable by the current user."
    exit 1
fi

# Parameter notes:
# - The verbose parameter is not quoted because it is optional.
# - The $@ parameter will pass the RV names along.

./runjava.sh $MAIN_CLASS ${VERBOSE} "${CFG_FILE}" ${COMMAND} -m "${MODE}" "$@"


