#!/bin/sh

# Copyright (C) 2010 Help/Systems LLC, All Rights Reserved
# Instructs an Agent to dump all active jobs to stdout.

logDebug="false"
MAIN_CLASS="com.helpsystems.enterprise.peer.AgentControl"

# Initialize a verbose parameter
# "verbose=" to turn off verbose mode
# "verbose=-v" to turn on verbose mode
verbose=

# initialize empty parm flags
CFG_FILE=agent.xml
COMMAND=getactivity
monitor=
jobs=
transfer=

# ------------------------------------------------------------------------
# Function: Log a debug message                                          -
# ------------------------------------------------------------------------
debugMsg() {
  if test "$logDebug" = "true"
  then 
    echo "DEBUG: $1"
  fi
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  echo ""
  echo "Help for: $0"
  echo ""
  echo "GETACTIVITY - Instructs an Agent to dump all active jobs to stdout."
  echo ""
  echo "Usage: $0 [-f config file]"
  echo "If no config file is provided, then the file agent.xml is used."
  echo ""
  echo "All info will be shown by default if no options are selected." 
  echo "  [-j]                  show active jobs"
  echo "  [-m]                  show active job monitors"
  echo "  [-t]                  show current file transfers"
  echo "  [-v]                  verbose mode"
  echo ""
  echo ""
  echo "  Exit Status:"
  echo "    0=Success"
  echo "    1=Failure"
  echo ""
}

# ------------------------------------------------------------------------
# Main Line Code.                                                        -
# ------------------------------------------------------------------------

if [ $# -eq 0 ]; then
  jobs=-j
  monitor=-m
  transfer=-t
fi

if [ $# -gt 0 -a "$1" = "--help" ]; then
  showHelpText
  exit 0
fi

#Check for config file first
if [ "$1" = "-f" ]; then
   if [ -z "$2" ]; then
     echo "Error: A configuration file must follow the -f parameter"
     exit 1
   fi
fi

while getopts hvf:jmt opt
  do
    case $opt in
    h)   showHelpText
         exit 0
         ;;
    v)   verbose=-v
         debugMsg "Setting on verbose mode"
         ;;
    f)   CFG_FILE=$OPTARG
         debugMsg "Config file: $CFG_FILE"
         ;;
    j)   jobs=-j
         debugMsg "Setting on jobs mode"
         ;;
    m)   monitor=-m
         debugMsg "Setting on monitor mode"
         ;;
    t)   transfer=-t
         debugMsg "Setting on transfer mode"
         ;;
    '?') echo "$0: invalid option -$opt" >&2
         showHelpText
         exit 1
         ;;
    esac
  done

debugMsg "----------------------------------------------------------------"
debugMsg "- Verbose flag  . . . . . . : ${verbose}"
debugMsg "- Using config file . . . . : ${CFG_FILE}"
debugMsg "- Sending Command . . . . . : ${COMMAND}"
debugMsg "- Jobs Flag . . . . . . . . : ${jobs}"
debugMsg "- Transfer Flag . . . . . . : ${transfer}"
debugMsg "- Monitor Flag. . . . . . . : ${monitor}"
debugMsg "----------------------------------------------------------------"

PATH=$PATH:/usr/ucb

WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}

if [ ! -f ${CFG_FILE} ]; then
    echo "The config file ${CFG_FILE} does not exist. Command failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "The config file $CFG_FILE is not readable by the current user."
    exit 1
fi

# Parameter notes:
# - The verbose parameter is not quoted because it is optional.
echo "About to dump all active jobs for the Agent with configuration file $CFG_FILE ..."
./runjava.sh $MAIN_CLASS ${verbose} "${CFG_FILE}" ${COMMAND} ${jobs} ${monitor} ${transfer}
