"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

require("bluebird-global");

var _lodash = _interopRequireWildcard(require("lodash"));

var _patternsUtils = require("../../tools/patterns-utils");

var _tokenizers = require("../language/tokenizers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

const debug = DEBUG('nlu').sub('entities');
const debugLists = debug.sub('lists');
const MIN_LENGTH_FUZZY_MATCH = 4;
const MIN_CONFIDENCE = 0.65;

const stripSpecialChars = str => str.replace(/[`~!@#$%^&*()_|+\-=?;:'",.<>\{\}\[\]\\\/]/gi, '');

class PatternExtractor {
  constructor(toolkit) {
    this.toolkit = toolkit;
  }

  async extractLists(input, lang, entityDefs) {
    const entities = (0, _lodash.flatten)((0, _lodash.flatten)((await Promise.mapSeries(entityDefs, async entityDef => {
      return await Promise.mapSeries(entityDef.occurences || [], occurence => this._extractEntitiesFromOccurence(input, lang, occurence, entityDef));
    }))));
    return _lodash.default.orderBy(entities, ['meta.confidence'], ['desc']);
  }

  async _extractEntitiesFromOccurence(input, lang, occurence, entityDef) {
    const tokens = await (0, _tokenizers.tokenize)(input, lang);
    const values = await Promise.mapSeries([occurence.name, ...occurence.synonyms], v => (0, _tokenizers.tokenize)(v, lang));
    const findings = [];
    let cur = 0;

    for (const tok of tokens) {
      cur = cur + input.substr(cur).indexOf(tok);
      let highest = 0;
      let extracted = '';
      let source = '';

      for (const val of values) {
        let partOfPhrase = tok;
        const occ = val.join('+');

        if (val.length > 1) {
          const _tokens = await (0, _tokenizers.tokenize)(input.substr(cur + partOfPhrase.length), lang);

          while (_tokens && _tokens.length && partOfPhrase.length < occ.length) {
            partOfPhrase += '+' + _tokens.shift();
          }
        }

        let distance = 0.0;

        if (entityDef.fuzzy && partOfPhrase.length > MIN_LENGTH_FUZZY_MATCH) {
          const d1 = this.toolkit.Strings.computeLevenshteinDistance(partOfPhrase, occ);
          const d2 = this.toolkit.Strings.computeJaroWinklerDistance(partOfPhrase, occ, {
            caseSensitive: true
          });
          distance = Math.min(d1, d2);
          const diffLen = Math.abs(partOfPhrase.length - occ.length);

          if (diffLen <= 3) {
            distance = Math.min(1, distance * (0.1 * (4 - diffLen) + 1));
          }
        } else {
          const strippedPop = stripSpecialChars(partOfPhrase.toLowerCase());
          const strippedOcc = stripSpecialChars(occ.toLowerCase());

          if (strippedPop.length && strippedOcc.length) {
            distance = strippedPop === strippedOcc ? 1 : 0;
          }
        } // if is closer OR if the match found is longer


        if (distance > highest || distance === highest && extracted.length < occ.length) {
          extracted = occ;
          highest = distance;
          source = input.substr(cur, partOfPhrase.length);
        }
      }

      const start = cur;
      const end = cur + source.length; // prevent adding substrings of an already matched, longer entity

      const hasBiggerMatch = findings.find(x => start >= x.meta.start && end <= x.meta.end);

      if (highest >= MIN_CONFIDENCE && !hasBiggerMatch) {
        debugLists('found list entity', {
          lang,
          occurence: occurence.name,
          input,
          extracted,
          confidence: highest,
          source
        });
        findings.push({
          name: entityDef.name,
          type: 'list',
          meta: {
            confidence: highest,
            // extrated with synonyme as patterns
            provider: 'native',
            source: source,
            start,
            end,
            raw: {}
          },
          data: {
            extras: {
              occurence: extracted
            },
            value: occurence.name,
            // cannonical value,
            unit: 'string'
          }
        });
      }
    }

    return findings;
  }

  async extractPatterns(input, entityDefs) {
    return (0, _lodash.flatMap)(entityDefs, entityDef => {
      try {
        const regex = new RegExp(entityDef.pattern);
        return (0, _patternsUtils.extractPattern)(input, regex).map(res => ({
          name: entityDef.name,
          type: entityDef.type,
          // pattern
          sensitive: entityDef.sensitive,
          meta: {
            confidence: 1,
            // pattern always has 1 confidence
            provider: 'native',
            source: res.value,
            start: res.sourceIndex,
            end: res.sourceIndex + res.value.length,
            raw: {}
          },
          data: {
            extras: {},
            value: res.value,
            unit: 'string'
          }
        }));
      } catch (error) {
        throw Error(`Pattern of entity ${entityDef.name} is invalid`);
      }
    });
  }

}

exports.default = PatternExtractor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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