"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DucklingEntityExtractor = void 0;

var _axios = _interopRequireDefault(require("axios"));

var _httpsProxyAgent = _interopRequireDefault(require("https-proxy-agent"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class DucklingEntityExtractor {
  constructor(logger) {
    this.logger = logger;
  }

  static async configure(enabled, url, logger) {
    if (enabled) {
      const proxyConfig = process['PROXY'] ? {
        httpsAgent: new _httpsProxyAgent.default(process['PROXY'])
      } : {};
      this.client = _axios.default.create({
        baseURL: url,
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded'
        },
        ...proxyConfig
      });
      const ducklingDisabledMsg = `, so it will be disabled.
For more informations (or if you want to self-host it), please check the docs at
https://botpress.io/docs/build/nlu/#system-entities
`;

      try {
        const {
          data
        } = await this.client.get('/');

        if (data !== 'quack!') {
          return logger.warn(`Bad response from Duckling server ${ducklingDisabledMsg}`);
        }

        this.enabled = true;
      } catch (err) {
        logger.attachError(err).warn(`Couldn't reach the Duckling server ${ducklingDisabledMsg}`);
      }
    }
  }

  async extract(text, lang) {
    if (!DucklingEntityExtractor.enabled) return [];

    try {
      const tz = this._getTz();

      const {
        data
      } = await DucklingEntityExtractor.client.post('/parse', `lang=${lang}&text=${text}&reftime=${Date.now()}&tz=${tz}`);

      if (!_lodash.default.isArray(data)) {
        throw new Error('Unexpected response from Duckling. Expected an array.');
      }

      return data.map(ent => ({
        name: ent.dim,
        type: 'system',
        meta: this._mapMeta(ent),
        data: this._mapBody(ent.dim, ent.value)
      }));
    } catch (err) {
      const error = err.response ? err.response.data : err;
      this.logger && this.logger.attachError(error).warn('[Native] error extracting duckling entities');
      return [];
    }
  }

  _getTz() {
    return Intl.DateTimeFormat().resolvedOptions().timeZone;
  }

  _mapMeta(DEntity) {
    return {
      confidence: 1,
      // rule based extraction
      provider: 'native',
      source: DEntity.body,
      start: DEntity.start,
      end: DEntity.end,
      raw: DEntity
    };
  }

  _mapBody(dimension, rawVal) {
    switch (dimension) {
      case 'duration':
        const normalized = rawVal.normalized;
        delete rawVal['normalized'];
        return { ...normalized,
          extras: rawVal
        };

      case 'quantity':
        return {
          value: rawVal.value,
          unit: rawVal.unit,
          extras: {
            product: rawVal.product
          }
        };

      case 'time':
        return {
          value: rawVal.value,
          unit: rawVal.grain,
          extras: rawVal.values.length ? rawVal.values : {}
        };

      default:
        return {
          extras: {},
          value: rawVal.value,
          unit: rawVal.unit
        };
    }
  }

}

exports.DucklingEntityExtractor = DucklingEntityExtractor;

_defineProperty(DucklingEntityExtractor, "enabled", void 0);

_defineProperty(DucklingEntityExtractor, "client", void 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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