#!/bin/sh

# Copyright (C) 2017 HelpSystems, LLC, All Rights Reserved
# Starts (and restarts) an Agent.

PATH=$PATH:/usr/ucb
PLATFORM=`uname`
WORKING_DIR=`dirname $0`

if [ -z "$has_setsid" ]; then
    has_setsid=yes
    export has_setsid
    if [ 'Linux' = $PLATFORM ]; then
        # ensure that we are running as a session leader
        exec setsid $0 $@
    else
        exec ${WORKING_DIR}/setpgrp.sh $0 $@
    fi
fi

#if [ `whoami` != "root" ]; then
#    echo This process must be started by root.
#    exit
#fi

cd ${WORKING_DIR}

if [ $# -eq 0 ]; then
    CONFIG_FILE=agent.xml
else
    if [ "$1" = "-h" -o "$1" = "--help" ]; then
        echo ""
        echo "Help for: $0"
        echo ""
	    echo "START - Connects to the Agent Service on the local machine and "
        echo "        instructs it to start the agent with the passed in"
        echo "        configuration file name."
        echo ""
        echo "Usage: $0 [-f config file]"
        echo "If no config file is provided, then the file agent.xml is used."
        echo ""
        exit
    fi
    if [ "$1" = "-f" ]; then
        if [ $# -eq 1 ]; then
           echo "Error: A configuration file must follow the -f parameter"
           exit 1
        fi
        shift
        CONFIG_FILE=$1
    else
	echo "Error: Valid parameters are -h --help or -f"
        exit 1
    fi
fi

if [ ! -f ${CONFIG_FILE} ]; then
    echo "The config file $CONFIG_FILE is missing. Startup unsuccessful."
    exit 1
fi

CURRENT_USER=`whoami` 2> /dev/null

if [ ! -w $CONFIG_FILE ]; then
   # The user starting the agent does not have permission
   # to write changes to the agent.xml file.
   echo "Unable to start the agent - the current user '${CURRENT_USER}' does not have"
   echo "the authority to write changes to the agent configuration file $CONFIG_FILE"
   exit 1
fi

if [ ! -r ${CONFIG_FILE} ]; then
    echo "The config file $CONFIG_FILE is not readable by the current user. Startup unsuccessful."
    exit 1
fi

PID_FILE=${CONFIG_FILE}.pid
OLD_PID=0
if [ -f $PID_FILE ]; then

    if [ -r $PID_FILE ]; then
        OLD_PID=`cat ${PID_FILE}`
        if [ $$ -eq $OLD_PID ]; then
            OLD_PID=0
        fi
    else
        echo "Unable to read file $PID_FILE."
        echo "Please check the file/directory permissions."
        exit 1
    fi

    if [ $OLD_PID -gt 0 ]; then
        PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`
        if [ 0 -ne "${PROCESS_COUNT}" ]; then
            echo "The Agent is currently running."
            exit
        fi
    fi
fi

PASSPHRASE_FILE=${CONFIG_FILE}.passphrase
if [ -f $PASSPHRASE_FILE ]; then
    rm -f $PASSPHRASE_FILE
    if [ $? -ne 0 ]; then
        echo "Unable to remove file $PASSPHRASE_FILE"
        exit 1
    fi
fi

touch $PASSPHRASE_FILE
if [ $? -ne 0 ]; then
    echo "Unable to create file $PASSPHRASE_FILE"
    exit 1
fi
chmod 600 $PASSPHRASE_FILE

# netstat command is deprecated on linux, and we should use ss instead
# I need to check if ss exists, however, because AIX still uses netstat
PORTSTAT_CMD='ss -an'
command -v ss >/dev/null 2>&1
if [ $? -eq 1 ]; then
    PORTSTAT_CMD='netstat -an'
fi    


LISTEN_PORT=`grep ListenPort ${CONFIG_FILE} | cut -f6 -d'"'`
if [ -z $LISTEN_PORT ]; then
    in_use=0
else
    in_use=`${PORTSTAT_CMD} | grep LISTEN | egrep "[^0-9]${LISTEN_PORT}" | wc -l`
fi

if [ 1 -eq "${in_use}" ]; then
    echo "The port ${LISTEN_PORT} is currently in use. Startup unsuccessful."
    exit 1
fi

# Import any custom environment settings the customer would like to use
if [ -r myenv.sh ]; then
    . ./myenv.sh "$@"
fi

LOGFILE_BASE=logs/`basename $CONFIG_FILE | cut -f1 -d.`
BOOTLOG=${LOGFILE_BASE}boot.log
LOGFILE=${LOGFILE_BASE}.log

touch $LOGFILE > /dev/null 2>&1
if [ $? -ne 0 ]; then
    echo "Unable to write to file $LOGFILE."
    echo "Please check the file/directory permissions."
    exit 1
fi

JAVA_OPTS="-DagentLog=${LOGFILE}"
export JAVA_OPTS


touch $PID_FILE > /dev/null 2>&1
if [ $? -ne 0 ]; then
    echo "Unable to write to file $PIDFILE."
    echo "Please check the file/directory permissions."
    exit 1
fi

echo $$ > $PID_FILE

if [ -z "$AUTO_UPDATE_DIR" ]; then
    AUTO_UPDATE_DIR=/tmp/$CURRENT_USER-schedule-auto-update
    if [ ${PLATFORM} = "HP-UX" -o ${PLATFORM} = "SunOS" ]; then
        # Bug 3813 - HP and Solaris JMVs use /var/tmp
        AUTO_UPDATE_DIR=/var${AUTO_UPDATE_DIR}
    fi
fi
export AUTO_UPDATE_DIR

if [ ! -d $AUTO_UPDATE_DIR ]; then
    mkdir -p $AUTO_UPDATE_DIR > /dev/null 2>&1
    if [ $? -ne 0 ] && [ ! -d $AUTO_UPDATE_DIR ]; then
        echo "Warning: Unable to create auto-update temp dir ${AUTO_UPDATE_DIR}"
        echo "         This agent cannot be updated by the Automate Schedule Server."
    fi
fi

if [ -z "$AUTO_UPDATE_SCRIPT" ]; then
    AUTO_UPDATE_SCRIPT=auto-update.sh
fi
export AUTO_UPDATE_SCRIPT

exitCode=99
while [ $exitCode -eq 99 ]; do
    # run the java process and pass all command line args to it
    ./runjava.sh com.helpsystems.enterprise.peer.StartAgent $CONFIG_FILE "$@" 
    
	# now take a look at the exit code that Java gave us.
	exitCode=$?

    if [ $exitCode -eq 97 ]; then
        if [ -f "$AUTO_UPDATE_DIR/${AUTO_UPDATE_SCRIPT}" ]; then
            rm -f $PID_FILE
            cd $AUTO_UPDATE_DIR
            chmod 755 ${AUTO_UPDATE_SCRIPT}
            exec ./${AUTO_UPDATE_SCRIPT} > update.log 2>&1
            echo "Unable to start the auto-update script ${AUTO_UPDATE_SCRIPT}" >> update.log
            exit 1
        else
            echo "Unable to update this agent, the script ${AUTO_UPDATE_SCRIPT} is missing."
        fi
    fi
    
	# if the exit code is 99 then the loop will continue, and the Java process will restart. If the 
	# Java process exists with anything other than 99, then the loop is done and the script exits.

done

rm -f $PID_FILE
rm -f $PASSPHRASE_FILE
