#!/bin/sh
#
# Copyright (C) 2017 HelpSystems, LLC, All Rights Reserved
#
# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  echo ""
  echo "                              import-cron"
  echo ""
  echo "             Import cron entries to the Automate Schedule Server"
  echo ""
  echo ""
  echo "Usage:  ./import-cron.sh [-f <configFile>][-h] <filePath>"
  echo ""
  echo "Mandatory Arguments:"
  echo "  <filepath> is the path to the crontab file."
  echo "  For example: ../crontab,  or"
  echo "               /etc/crontab, or"
  echo "               cron/johndoe"
  echo ""
  echo "Optional Arguments:"
  echo "  -f       If -f <configFile> is not provided, then the default configuration" 
  echo "           in the agent.xml file is used."
  echo ""
  echo "  -h       Displays this help text."
  echo ""
  echo "Only the root user is allowed to import crontab files to Automate Schedule"
  echo "to ensure full access to the content of the files as well as resources "
  echo "associated with cron entries."
  echo ""
  
  if  test `uname -s` = "Linux"
  then
    echo "The default location of user crontab files for Linux varies by distribution."
    echo "   Redhat/Fedora:  '/var/spool/cron/'"
    echo "   Debian . . . :  '/var/spool/cron/crontabs/'"
    echo "   Suse . . . . :  '/var/spool/cron/tabs/'"
  fi

  if  test `uname -s` = "AIX"
  then
    echo "The default location of user crontab files for AIX is '/var/spool/cron/crontabs/'"
  fi

  if  test `uname -s` = "SunOS"
  then
    echo "Files and directories typically associated with cron on SunOS:"
    echo "      /etc/cron.d                 main cron directory"
    echo "      /etc/cron.d/cron.allow      list of allowed users"
    echo "      /etc/default/cron           contains cron default settings"
    echo "      /etc/cron.d/cron.deny       list of denied users"
    echo "      /var/cron/log               accounting information"
    echo "      /var/spool/cron/crontabs    Directory containing the crontab files"
    echo ""
    echo "The default location of user crontab files for SunOS is '/var/spool/cron/crontabs/'"
  fi

  if  test `uname -s` = "HP-UX"
  then
    echo "Files and directories typically associated with cron on HP-UX:"
    echo "      /var/adm/cron                 Main cron directory"
    echo "      /var/adm/cron/cron.allow      List of allowed users"
    echo "      /var/adm/cron/cron.deny       List of denied users"
    echo "      /var/adm/cron/log             Accounting information"
    echo "      /var/spool/cron/crontabs      Directory containing the crontab files"
  fi
  echo ""
  echo "(C) Copyright HelpSystems, LLC, 2017"
  echo "    All Rights Reserved"
  echo "    Licensed Material, Property of HelpSystems, LLC"
  echo ""
}

if  test `uname -s` = "SunOS"
then
    PATH=$PATH:/usr/ucb
    export PATH
fi


filePath=""
MAIN_CLASS="com.helpsystems.enterprise.peer.AgentControl"

config_file=agent.xml
COMMAND=IMPORTCRON

SHIFT_COUNT=0;
while getopts ":f:h" opt;
do
    case $opt in
        f ) config_file=$OPTARG
            SHIFT_COUNT=`expr $SHIFT_COUNT + 2`;;
        h ) showHelpText
            exit 0;;
        \? ) echo ""
             echo "Error: The options you specified are not valid."
             echo ""
             showHelpText
             exit 1
    esac
done

shift $SHIFT_COUNT
filePath=$@

# ------------------------------------------------------------------------
# Main Line Code.
# ------------------------------------------------------------------------
#case $filePath in 
#*\ *) 
#    echo ""
#    echo "Error: The file name you specified is not valid."
#    echo ""
#    showHelpText
#    exit 1
#    ;;
#esac

if test "$filePath" = ""
then
    echo ""
    echo "Error: Required file path parameter with the name of the crontab file was not specified."
    echo ""
    showHelpText
    exit 1
fi

if [ ! -f "${filePath}" ]; then
    echo "Error: The specified crontab file ${filePath} does not exist. Cron import failed."
    exit 1
fi

CURRENT_DIR=`pwd`
WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}
WORKING_DIR=`pwd`
CFG_FILE=$WORKING_DIR/${config_file}

if [ ! -f ${CFG_FILE} ]; then
    echo "Error: The specified configuration file ${CFG_FILE} does not exist. Cron import failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "Error: The configuration file $CFG_FILE is not readable by the current user."
    exit 1
fi

cd $CURRENT_DIR
cd "`dirname "$filePath"`"
truePath=`pwd`/`basename "$filePath"`
cd ${WORKING_DIR}

echo "Executing Command: $0 $@"
echo ""
echo "Import Environment:                                  "
echo "  Crontab . . . . . . . . . . :  $truePath           "
echo "  Configuration File Name . . :  $CFG_FILE           "
echo "  Current directory . . . . . :  $CURRENT_DIR        "
echo "  Working directory . . . . . :  $WORKING_DIR        "

./runjava.sh $MAIN_CLASS "$CFG_FILE" $COMMAND "$truePath"
cd $CURRENT_DIR

