#!/bin/sh

# Copyright (C) 2009 Help/Systems LLC, All Rights Reserved
# Allows user to send a FTP from or to an Agent.

logDebug="false"
MAIN_CLASS="com.helpsystems.enterprise.peer.FTPWorker"

# Initialize a verbose parameter
# "verbose=" to turn off verbose mode
# "verbose=-v" to turn on verbose mode
verbose=

# initialize empty parm flags
CFG_FILE=agent.xml
COMMAND=ftp
parm_i=
parm_u=
parm_p=
parm_s=
parm_t=
parm_x=
parm_m=:BINARY
parm_o=:PUT
parm_r=:YES


# ------------------------------------------------------------------------
# Function: Log an error message                                         -
# ------------------------------------------------------------------------
logError() {
  printf "$1\n">&2
}

# ------------------------------------------------------------------------
# Function: Log an info message                                         -
# ------------------------------------------------------------------------
logInfo() {
  printf "$1\n"
}

# ------------------------------------------------------------------------
# Function: Log a debug message                                          -
# ------------------------------------------------------------------------
debugMsg() {
  if test "$logDebug" = "true"
  then 
    logInfo "DEBUG: $1"
  fi
}

# ------------------------------------------------------------------------
# Function: Show Usage Text                                              -
# ------------------------------------------------------------------------
showUsageText() {
  logInfo "FTP - Allows user to put or get files from another system,"
  logInfo "      to or from this Agent."
  logInfo ""
  logInfo "Usage: $0 [-f <config_file>]

          [-f config_file]         the Agent configuration file to use.
           When a configuration file is not specified,
           the agent.xml file is used.
          [-h]                     display help information
          -i:<host name>           the Host you want to connect to
          -u:<user name>           the User you want to connect as
          -p:<encrypted password>  the encrypted password,
                                   encrypted by our processes
          -t:<target file>         the properly quoted target file
          -s:<source file>         the properly quoted source file
          [-m:<BINARY[TEXT]>]      the mode BINARY or TEXT, default is BINARY
          [-o:<PUT[GET]>]          the operation PUT or GET, default is PUT
          [-r:<YES or NO>]         the replace operation YES or NO,
                                   default is YES
          [-v]                     the verbose mode
          [-x]                     the use SSL secured mode"
  logInfo ""
}

# ------------------------------------------------------------------------
# Function: Show Help Text                                               -
# ------------------------------------------------------------------------
showHelpText() {
  logInfo ""
  logInfo "Help for: $0"
  logInfo ""
  showUsageText
  logInfo ""
  logInfo "   To get a file on a host named SERVER1:"
  logInfo ""
  logInfo "     Usage: $0 -i:^<host name^> -u:USERNAME -p:^<encyrpted password^> "
  logInfo "     -t:^<target path^> -s:^<source path^> -o:^<GET^> "
  logInfo "     [-m:^<BINARY^>] [-r:^<YES^>]"
  logInfo ""
  logInfo ""
  logInfo "     Example: $0 -i:SERVER1 -u:SOMEBODY -p:01020304050607080910 -o:GET"
  logInfo "     -s:\"//my directory/my sub/myfile.prf\" -t:\"C:\my ftp files\myfile.prf\""
  logInfo "     where SOMEBODY is your user name"
  logInfo ""
  logInfo ""
  logInfo "   To put a file on a host named SERVER1:"
  logInfo ""
  logInfo "     Usage: $0 -i:^<host name^> -u:USERNAME -p:^<encyrpted password^> "
  logInfo "     -t:^<target path^> -s:^<source path^> -o:^<PUT^> [-m:^<BINARY^>]"
  logInfo "     [-r:^<YES^>]"
  logInfo ""
  logInfo "     Example: $0 -i:SERVER1 -u:SOMEBODY -p:01020304050607080910 -o:PUT"
  logInfo "          -t:\"//my directory/my sub/myfile.prf\" -s \"/my ftp files/myfile.prf\""
  logInfo "         where SOMEBODY is your user name "
  logInfo ""
  logInfo "  Exit Status:"
  logInfo "    0=Success"
  logInfo "    1=Failure"
  logInfo ""
}

# ------------------------------------------------------------------------
# Main Line Code.                                                        -
# ------------------------------------------------------------------------

if [ $# -eq 0 ]; then
  logError "Host information and file parameters are required."
  showUsageText
  exit 1
fi

if [ "$1" = "--help" ]; then
  showHelpText
  exit 0
fi

#Check for config file first
if [ "$1" = "-f" ]; then
   if [ -z "$2" ]; then
     echo "Error: A configuration file must follow the -f parameter"
     exit 1
   fi
fi

while getopts :hvf:i:u:p:t:s:m:o:r:x: opt
  do
    case $opt in
    h)   showHelpText
         exit 0
         ;;
    v)   verbose=-v
         debugMsg "Setting on verbose mode"
         ;;
    f)   CFG_FILE=$OPTARG
         debugMsg "Config file: $CFG_FILE"
         ;;
    i)   parm_i=$OPTARG
         debugMsg "parm_i: $parm_i"
         ;;
    u)   parm_u=$OPTARG 
         debugMsg "parm_u: $parm_u"
         ;;
    p)   parm_p=$OPTARG 
         debugMsg "parm_p: $parm_p"
         ;;
    t)   parm_t=$OPTARG 
         debugMsg "parm_t: $parm_t"
         ;;
    s)   parm_s=$OPTARG 
         debugMsg "parm_s: $parm_s"
         ;;
    m)   parm_m=$OPTARG 
         debugMsg "parm_m: $parm_m"
         ;;
    o)   parm_o=$OPTARG 
         debugMsg "parm_o: $parm_o"
         ;;
    r)   parm_r=$OPTARG 
         debugMsg "parm_r: $parm_r"
         ;;
    x)   parm_x=-x
         debugMsg "parm_x: $parm_x"
         ;;
    '?') logError "$0: invalid option -$OPTARG"
         showUsageText
         exit 1
         ;;
    esac
  done

debugMsg "----------------------------------------------------------------"
debugMsg "- Verbose flag  . . . . . . : ${verbose}"
debugMsg "- Secure flag . . . . . . . : ${parm_x}"
debugMsg "- Using config file . . . . : ${CFG_FILE}"
debugMsg "- Sending Command . . . . . : ${COMMAND}"
debugMsg "- I-Parmateter  . . . . . . : ${parm_i}"
debugMsg "- U-Parameter . . . . . . . : ${parm_u}"
debugMsg "- P-Parameter . . . . . . . : ${parm_p}"
debugMsg "- T-Parameter . . . . . . . : ${parm_t}"
debugMsg "- S-Parameter . . . . . . . : ${parm_s}"
debugMsg "- M-Parameter . . . . . . . : ${parm_m}"
debugMsg "- O-Parameter . . . . . . . : ${parm_o}"
debugMsg "- R-Parameter . . . . . . . : ${parm_r}"
debugMsg "----------------------------------------------------------------"

PATH=$PATH:/usr/ucb

WORKING_DIR=`dirname $0`
cd ${WORKING_DIR}

if [ ! -f ${CFG_FILE} ]; then
    logError "The config file ${CFG_FILE} does not exist. Command failed."
    exit 1
fi

if [ ! -r ${CFG_FILE} ]; then
    echo "The config file $CFG_FILE is not readable by the current user."
    exit 1
fi

# Parameter notes:
# - There are double quotes around the parms that could contain embedded blanks.
# - The verbose parameter is not quoted because it is optional.
# - We strip the secure mode off since there is only one
echo "About to FTP using the Agent with configuration file $CFG_FILE ..."
./runjava.sh $MAIN_CLASS "${CFG_FILE}" -i"${parm_i}" -u"${parm_u}" -p"${parm_p}" -s"${parm_s}" -t"${parm_t}" -m"${parm_m}" -o"${parm_o}" -r"${parm_r}" ${parm_x} ${verbose}
